/*
 * Copyright European Commission's
 * Taxation and Customs Union Directorate-General (DG TAXUD).
 */
package eu.europa.ec.taxud.cesop.readers;

import javax.xml.namespace.QName;
import javax.xml.stream.XMLStreamException;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.time.format.DateTimeParseException;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Optional;

import org.codehaus.stax2.validation.XMLValidationSchema;

import eu.europa.ec.taxud.cesop.domain.MessageTypeEnum;
import eu.europa.ec.taxud.cesop.domain.MessageTypeIndicEnum;
import eu.europa.ec.taxud.cesop.domain.PaymentDataMsgPartContentType;
import eu.europa.ec.taxud.cesop.domain.XmlMessageSpec;
import eu.europa.ec.taxud.cesop.domain.XmlPaymentDataMsgPart;
import eu.europa.ec.taxud.cesop.domain.XmlPsp;
import eu.europa.ec.taxud.cesop.domain.XmlReportedPayee;
import eu.europa.ec.taxud.cesop.domain.XmlReportedTransaction;
import eu.europa.ec.taxud.cesop.utils.ValidationConstants.KEY;
import eu.europa.ec.taxud.cesop.utils.ValidationConstants.XML;
import eu.europa.ec.taxud.cesop.xsd.XsdSchema;

import static eu.europa.ec.taxud.cesop.utils.ValidationErrorUtils.XML_DATE_TIME_FORMATTER;
import static eu.europa.ec.taxud.cesop.utils.ValidationErrorUtils.convertGreece;

/**
 * Utils class reading a PSP XML file.
 */
public class PspXmlReader implements IPspXmlReader {

    private final int maxTransactionsInPart;
    private final XmlMessageSpec xmlMessageSpec;
    private final XmlPsp xmlReportingPsp;
    private final long estimatedContentLength;
    private final CesopXmlReader xmlReader;
    private XmlPaymentDataMsgPart paymentDataMsgPart;
    private ReportedPayeeXmlReader reportedPayeeXmlReader;
    private long partNumber;
    private boolean emptyTransactionsCurrentPayee = true;
    private boolean moveCursorToNextReportedPayee = true;

    /**
     * Constructor.
     *
     * @param inputStream           the input stream. The input stream is not closed by {@link PspXmlReader}.
     * @param estimatedSize         estimated content length in bytes, negative means that estimation is not available
     * @param maxTransactionsInPart the max number of transactions to be included in a payment data msg part
     * @param validateXsd           true if validation against {@link XsdSchema} needed
     * @throws XMLStreamException in case of error while processing the XML content
     */
    public PspXmlReader(final InputStream inputStream, final long estimatedSize, final int maxTransactionsInPart,
            boolean validateXsd) throws XMLStreamException {
        BufferedInputStream buffer = new BufferedInputStream(inputStream);
        this.xmlReader = new CesopXmlReader(buffer, validateXsd ? lookupXsd(buffer) : null);
        this.estimatedContentLength = estimatedSize;
        this.maxTransactionsInPart = maxTransactionsInPart;
        this.xmlMessageSpec = this.initXmlMessageSpec();
        if (xmlReader.positionCursorOnStartElement(XML.PAYMENT_DATA_BODY_QNAME)) {
            this.xmlReportingPsp = this.parsePsp();
        } else {
            this.xmlReportingPsp = null;
        }
    }

    private static XMLValidationSchema lookupXsd(BufferedInputStream inputStream) throws XMLStreamException {
        inputStream.mark(8192); // default buffer size
        XmlStreamReaderWrapper versionReader = new XmlStreamReaderWrapper(inputStream, null);

        if (!versionReader.goToNextStartElement(XML.CESOP_QNAME)) {
            throw UnknownXsdVersionException.unparsableVersion();
        }
        Map<String, String> attrs = versionReader.getAttributes();
        String version = attrs.get("version");
        if (version == null) {
            throw UnknownXsdVersionException.unparsableVersion();
        }
        try {
            inputStream.reset();
        } catch (IOException e) {
            throw new XMLStreamException(e);
        }
        try {
            // normalize version: 4 => 4.00, 4.0010 => 4.001
            DecimalFormat versionFormat = new DecimalFormat("0.00", DecimalFormatSymbols.getInstance(Locale.ROOT));
            versionFormat.setMinimumFractionDigits(2);
            versionFormat.setMaximumFractionDigits(40);
            version = versionFormat.format(new BigDecimal(version));
        } catch (NumberFormatException e) {
            throw UnknownXsdVersionException.unsupportedVersion(version);
        }
        XMLValidationSchema result = XsdSchema.VERSION_2_XSD.get(version);
        if (result == null) {
            throw UnknownXsdVersionException.unsupportedVersion(version);
        }
        return result;
    }

    @Override
    public long getEstimatedContentSize() {
        return this.estimatedContentLength;
    }

    private XmlMessageSpec initXmlMessageSpec() throws XMLStreamException {
        final XmlMessageSpec messageSpecXML = new XmlMessageSpec();

        xmlReader.positionCursorOnStartElement(XML.CESOP_QNAME);
        Map<String, String> attrs = xmlReader.getXmlStreamReaderWrapper().getAttributes();
        messageSpecXML.setXsdVersion(attrs.get(XML.ATTRIBUTE_NAME_VERSION));

        final Map<String, String> transmittingCountryValues = xmlReader.readNextTagIntoMap(XML.TRANSMITTING_COUNTRY_QNAME);
        messageSpecXML.setTransmittingCountry(convertGreece(transmittingCountryValues.get(KEY.TRANSMITTING_COUNTRY_KEY)));

        final Map<String, String> messageTypeValues = xmlReader.readNextTagIntoMap(XML.MESSAGE_TYPE_QNAME);
        messageSpecXML.setMessageType(MessageTypeEnum.findByLabel(messageTypeValues.get(KEY.MESSAGE_TYPE_KEY)));

        final Map<String, String> messageTypeIndicValues = xmlReader.readNextTagIntoMap(XML.MESSAGE_TYPE_INDIC_QNAME);
        messageSpecXML.setMessageTypeIndic(MessageTypeIndicEnum.valueOf(messageTypeIndicValues.get(KEY.MESSAGE_TYPE_INDIC_KEY)));

        final Map<String, String> messageRefIdValues = xmlReader.readNextTagIntoMap(XML.MESSAGE_REF_ID_QNAME);
        messageSpecXML.setMessageRefId(messageRefIdValues.get(KEY.MESSAGE_REF_ID_KEY));

        final Optional<Map<String, String>> correlationMessageRefIdValuesOpt = xmlReader.readNextTagIfEquals(XML.CORR_MESSAGE_REF_ID_QNAME);
        correlationMessageRefIdValuesOpt.ifPresent(correlationMessageRefIdValues -> messageSpecXML.setCorrMessageRefId(correlationMessageRefIdValues.get(KEY.CORR_MESSAGE_REF_ID_KEY)));

        if (xmlReader.positionCursorOnStartElement() && xmlReader.getStartElementName().equals(XML.SENDING_PSP_QNAME)) {
            messageSpecXML.setSendingPsp(this.parsePsp());
        }

        final Map<String, String> quarterValues = xmlReader.readNextTagIntoMap(XML.QUARTER_QNAME);
        final String periodQuarter = quarterValues.get(KEY.QUARTER_KEY);
        final Map<String, String> yearValues = xmlReader.readNextTagIntoMap(XML.YEAR_QNAME);
        final String periodYear = yearValues.get(KEY.YEAR_KEY);
        messageSpecXML.setReportingPeriod(Integer.parseInt(periodYear + periodQuarter));

        final Map<String, String> timestampValues = xmlReader.readNextTagIntoMap(XML.TIMESTAMP_QNAME);
        messageSpecXML.setTimestamp(timestampValues.get(KEY.TIMESTAMP_KEY));

        // Validate timestamp
        try {
            XML_DATE_TIME_FORMATTER.parse(messageSpecXML.getTimestamp());
        } catch (DateTimeParseException e) {
            throw new CesopParsingException("Error while reading the XML file: " + e.getMessage(), e);
        }

        return messageSpecXML;
    }

    private XmlPsp parsePsp() throws XMLStreamException {
        final Map<String, String> reportingPspMap = xmlReader.readNextTagIntoMap(XML.PSP_ID_QNAME);
        final String pspIdType = reportingPspMap.get(KEY.PSP_ID_TYPE_KEY);
        final String pspIdOther = reportingPspMap.get(KEY.PSP_ID_OTHER_KEY);
        final String pspId = reportingPspMap.get(KEY.PSP_ID_KEY);
        final XmlPsp reportingPspXML = new XmlPsp(pspIdType, pspId, pspIdOther);
        Optional<Map<String, String>> valuesMap;
        while ((valuesMap = xmlReader.readNextTagIfEquals(XML.NAME_QNAME)).isPresent()) {
            final String pspNameType = valuesMap.get().get(KEY.NAME_TYPE_KEY);
            final String pspNameOther = valuesMap.get().get(KEY.NAME_OTHER_KEY);
            final String pspName = valuesMap.get().get(KEY.NAME_KEY);
            reportingPspXML.addName(pspNameType, pspName, pspNameOther);
        }
        return reportingPspXML;
    }

    @Override
    public boolean hasNext() {
        try {
            if (this.paymentDataMsgPart == null) {
                this.paymentDataMsgPart = this.createNextPaymentDataMsgPart();
            }
            return this.paymentDataMsgPart != null;
        } catch (final RuntimeException e) {
            throw e;
        } catch (final Exception e) {
            throw new CesopParsingException("Error while reading the XML file: " + e.getMessage(), e);
        }
    }

    @Override
    public XmlPaymentDataMsgPart next() {
        if (this.paymentDataMsgPart != null) {
            final XmlPaymentDataMsgPart result = this.paymentDataMsgPart;
            this.paymentDataMsgPart = null;
            return result;
        }
        if (this.hasNext()) {
            return this.next();
        }
        throw new NoSuchElementException();
    }

    private XmlPaymentDataMsgPart createNextPaymentDataMsgPart() throws XMLStreamException {
        if (!this.moveCursorToNextReportedPayee) {
            this.moveCursorToNextReportedPayee = true;
            return this.createReportedPayeePart();
        }
        while (xmlReader.positionCursorOnStartElement()) {
            final QName qName = xmlReader.getStartElementName();
            if (qName.equals(XML.REPORTED_PAYEE_QNAME)) {
                this.emptyTransactionsCurrentPayee = true;
                this.moveCursorToNextReportedPayee = true;
                this.initializeReportedPayeePart();
            } else if (qName.equals(XML.REPORTED_TRANSACTION_QNAME)) {
                this.emptyTransactionsCurrentPayee = false;
                return this.createReportedTransactionPart(false);//
            } else if (this.reportedPayeeXmlReader != null) {
                // <Representative> tag encountered
                if (this.emptyTransactionsCurrentPayee) {
                    /* if no <ReportedTransaction> encountered before (in case of a reported payee deletion),
                    we do still need to create a payment data msg part with content type 'T' for the ingestion to work
                    After that, we need to stop the cursor in order to create the reported payee part (payment data msg 'P')
                    before moving to the next <ReportedPayee>.
                     */
                    this.moveCursorToNextReportedPayee = false;
                    return this.createReportedTransactionPart(true);
                } else {
                    return this.createReportedPayeePart();
                }
            }
        }
        return null;
    }

    private void initializeReportedPayeePart() throws XMLStreamException {
        this.partNumber = 1;
        this.reportedPayeeXmlReader = new ReportedPayeeXmlReader(xmlReader);
        this.reportedPayeeXmlReader.parseFirstPart();
    }

    private XmlPaymentDataMsgPart createReportedPayeePart() throws XMLStreamException {
        try {
            this.reportedPayeeXmlReader.parseSecondPart();
            final XmlReportedPayee xmlReportedPayee = this.reportedPayeeXmlReader.getXmlReportedPayee();
            final XmlPaymentDataMsgPart paymentDataPart = new XmlPaymentDataMsgPart();
            paymentDataPart.setPartNumber(1L);
            paymentDataPart.setContentType(PaymentDataMsgPartContentType.REPORTED_PAYEE);
            paymentDataPart.setXmlReportedPayee(xmlReportedPayee);
            return paymentDataPart;
        } finally {
            this.reportedPayeeXmlReader = null;
        }
    }

    private XmlPaymentDataMsgPart createReportedTransactionPart(final boolean emptyTransactions) throws
            XMLStreamException {
        final XmlReportedPayee xmlReportedPayee = this.reportedPayeeXmlReader.getXmlReportedPayee();
        final ReportedTransactionXmlReader reportedTransactionXmlReader = new ReportedTransactionXmlReader(xmlReader,
                xmlReportedPayee.getOtherPaymentMethods(), xmlReportedPayee.getOtherPspRoles());
        final List<XmlReportedTransaction> xmlReportedTransactions = new LinkedList<>();
        if (!emptyTransactions) {
            for (int i = 0; i < this.maxTransactionsInPart; i++) {
                final XmlReportedTransaction xmlReportedTransaction = reportedTransactionXmlReader.parse();
                xmlReportedTransactions.add(xmlReportedTransaction);
                if (!xmlReader.positionCursorOnStartElement() || !xmlReader.getStartElementName().equals(XML.REPORTED_TRANSACTION_QNAME)) {
                    xmlReader.getXmlStreamReaderWrapper().markAsPeek();
                    break;
                }
            }
        } else {
            xmlReader.getXmlStreamReaderWrapper().markAsPeek();
        }
        final XmlPaymentDataMsgPart paymentDataPart = new XmlPaymentDataMsgPart();
        paymentDataPart.setPartNumber(++this.partNumber);
        paymentDataPart.setContentType(PaymentDataMsgPartContentType.REPORTED_TRANSACTIONS);
        paymentDataPart.setXmlReportedPayee(xmlReportedPayee);
        paymentDataPart.setXmlReportedTransactions(xmlReportedTransactions);
        return paymentDataPart;
    }

    @Override
    public XmlMessageSpec getXmlMessageSpec() {
        return this.xmlMessageSpec;
    }

    @Override
    public XmlPsp getXmlReportingPsp() {
        return this.xmlReportingPsp;
    }

    @Override
    public void close() throws Exception {
        this.xmlReader.close();
    }
}
